<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'WPDesk_Flexible_Shipping_SaaS_Shipping_Service' ) ) {

	/**
	 * Class WPDesk_Flexible_Shipping_SaaS_Shipping_Service
	 */
	class WPDesk_Flexible_Shipping_SaaS_Shipping_Service {

		const INTEGRATION_PREFIX = 'flexible_shipping_saas_';

		const FIELD_TYPE_CONNECTION_STATUS = 'connection_status';

		const FIELD_TITLE             = 'title';
		const FIELD_LABEL             = 'label';
		const FIELD_DESC              = 'desc';
		const FIELD_DESCRIPTION       = 'description';
		const FIELD_TOOLTIP           = 'tooltip';
		const FIELD_DESC_TIP          = 'desc_tip';
		const FIELD_CUSTOM_ATTRIBUTES = 'custom_attributes';
		const FIELD_TYPE              = 'type';
		const FIELD_REQUIRED          = 'required';
		const FIELD_SELECT_ADDITIONAL = 'select-additional';
		const FIELD_OPTIONS           = 'options';

		const TYPE_TEXT            = 'text';
		const TYPE_CHECKBOX        = 'checkbox';
		const TYPE_SELECT          = 'select';
		const TYPE_CUSTOM_SERVICES = 'custom_services';

		const WOOCOMMERCE_SETTINGS_SHIPPING_URL = 'admin.php?page=wc-settings&tab=shipping';

		/**
		 * SaaS connection.
		 *
		 * @var WPDesk_Flexible_Shipping_SaaS_Connection
		 */
		private $saas_connection;

		/**
		 * ID.
		 *
		 * @var int
		 */
		private $id;

		/**
		 * Name.
		 *
		 * @var string
		 */
		private $name;

		/**
		 * Description.
		 *
		 * @var string
		 */
		private $description;

		/**
		 * Promoted.
		 *
		 * @var bool
		 */
		private $promoted;

		/**
		 * Settings fields.
		 *
		 * @var array
		 */
		private $settings_fields;

		/**
		 * Definition of return values generated by settings.
		 *
		 * @var array
		 */
		private $settings_fields_values;

		/**
		 * Definition of request fields.
		 *
		 * @var array
		 */
		private $request_fields;

		/**
		 * Capabilities.
		 *
		 * @var array
		 */
		private $capabilities;

		/**
		 * Shipping method.
		 *
		 * @var null|WPDesk_Flexible_Shipping_SaaS_Shipping_Service_Settings
		 */
		private $shipping_method = null;

		/**
		 * Links.
		 *
		 * @var WPDesk_Flexible_Shipping_SaaS_Platform_Links
		 */
		private $saas_platform_links;

		/**
		 * Integration id.
		 *
		 * @var string
		 */
		private $integration_id;

		/**
		 * WPDesk_Flexible_Shipping_SaaS_Shipping_Service constructor.
		 *
		 * @param WPDesk_Flexible_Shipping_SaaS_Connection     $saas_connection Saas connection.
		 * @param WPDesk_Flexible_Shipping_SaaS_Platform_Links $saas_platform_links Links.
		 * @param int                                          $id Service ID.
		 * @param string                                       $name Service Name.
		 * @param string                                       $description Service description.
		 * @param bool                                         $promoted Promoted.
		 * @param array                                        $settings_fields Settings fields.
		 * @param array                                        $settings_fields_values Definition of return values generated by settings.
		 * @param array                                        $request_fields Request fields.
		 * @param array                                        $capabilities Capabilities.
		 */
		public function __construct(
			WPDesk_Flexible_Shipping_SaaS_Connection $saas_connection,
			WPDesk_Flexible_Shipping_SaaS_Platform_Links $saas_platform_links,
			$id,
			$name,
			$description,
			$promoted,
			array $settings_fields = array(),
			$settings_fields_values = array(),
			$request_fields = array(),
			$capabilities = array()
		) {
			$this->saas_connection        = $saas_connection;
			$this->saas_platform_links    = $saas_platform_links;
			$this->id                     = $id;
			$this->name                   = $name;
			$this->description            = $description;
			$this->promoted               = $promoted;
			$this->settings_fields        = $settings_fields;
			$this->settings_fields_values = $settings_fields_values;
			$this->request_fields         = $request_fields;
			$this->capabilities           = $capabilities;

			$this->integration_id = self::INTEGRATION_PREFIX . $id;
		}

		/**
		 * Static creator form \WPDesk\SaasPlatformClient\Model\ShippingService.
		 *
		 * @param \WPDesk_Flexible_Shipping_SaaS_Connection        $saas_connection SaaS connection.
		 * @param \WPDesk_Flexible_Shipping_SaaS_Platform_Links    $saas_platform_links Links.
		 * @param \WPDesk\SaasPlatformClient\Model\ShippingService $platform_service Service from platform.
		 * @return WPDesk_Flexible_Shipping_SaaS_Shipping_Service
		 */
		public static function create_from_platform_service(
			\WPDesk_Flexible_Shipping_SaaS_Connection $saas_connection,
			\WPDesk_Flexible_Shipping_SaaS_Platform_Links $saas_platform_links,
			\WPDesk\SaasPlatformClient\Model\ShippingService $platform_service
		) {
			return new self(
				$saas_connection,
				$saas_platform_links,
				$platform_service->getId(),
				$platform_service->getName(),
				$platform_service->getDescription(),
				$platform_service->getPromoted(),
				$platform_service->getConnectionSettingsDefinitionJson(),
				$platform_service->getConnectionSettingsValuesJsonSchema(),
				$platform_service->getRequestFieldsDefinitionJson(),
				$platform_service->getCapabilitiesJson()
			);
		}

		/**
		 * Get ID.
		 *
		 * @return int
		 */
		public function get_id() {
			return $this->id;
		}

		/**
		 * Get name.
		 *
		 * @return string
		 */
		public function get_name() {
			return $this->name;
		}

		/**
		 * Get description.
		 *
		 * @return string
		 */
		public function get_description() {
			return $this->description;
		}

		/**
		 * Get promoted.
		 *
		 * @return bool
		 */
		public function get_promoted() {
			return $this->promoted;
		}

		/**
		 * Get settings fields.
		 *
		 * @return array
		 */
		public function get_settings_fields() {
			return $this->settings_fields;
		}

		/**
		 * Get settings fields values.
		 *
		 * @return array
		 */
		public function get_settings_fields_values() {
			return $this->settings_fields_values;
		}

		/**
		 * Get request fields.
		 *
		 * @return array
		 */
		public function get_request_fields() {
			return $this->request_fields;
		}

		/**
		 * Get capabilities.
		 *
		 * @return array
		 */
		public function get_capabilities() {
			return $this->capabilities;
		}

		/**
		 * Get integration id.
		 *
		 * @return string
		 */
		public function get_integration_id() {
			return $this->integration_id;
		}

		/**
		 * Is capability?
		 *
		 * @param string $capability_name Capability name.
		 *
		 * @return bool
		 */
		public function supports( $capability_name ) {
			$capability = $this->get_capability( $capability_name );
			if ( isset( $this->capabilities[ $capability_name ] ) && $this->capabilities[ $capability_name ] ) {
				return true;
			}
			return false;
		}

		/**
		 * Get capability.
		 *
		 * @param string $capability_name Capability name.
		 *
		 * @return string|null
		 */
		public function get_capability( $capability_name ) {
			if ( isset( $this->capabilities[ $capability_name ] ) ) {
				return $this->capabilities[ $capability_name ];
			}
			return null;
		}

		/**
		 * Prapare section heading field.
		 *
		 * @param array $section Section.
		 *
		 * @return array
		 */
		private function prepare_section_heading_field( array $section ) {
			$section_heading = array(
				self::FIELD_TYPE => 'title',
				'title'          => $section['title'],
			);
			if ( isset( $section[ self::FIELD_DESCRIPTION ] ) ) {
				$section_heading['description'] = $section[ self::FIELD_DESCRIPTION ];
			}

			return $section_heading;
		}

		/**
		 * Prepare default field value.
		 *
		 * @param array $field Field from SaaS platform.
		 * @param array $settings_field WooCommerce settings field.
		 *
		 * @return array
		 */
		private function prepare_default_field_value( array $field, array $settings_field ) {
			if ( isset( $field['default'] ) ) {
				$settings_field['default'] = $field['default'];
			}
			$default_woocommerce_id = 'default-woocommerce';
			if ( isset( $field[ $default_woocommerce_id ] ) ) {
				if ( ! isset( $settings_field['default'] ) ) {
					$settings_field['default'] = get_option( $field[ $default_woocommerce_id ], '' );
				}
			}
			$deafault_commpute = 'default-compute';
			if ( isset( $field[ $deafault_commpute ] ) ) {
				if ( ! isset( $settings_field['default'] ) ) {
					$default_compute_handler   = new WPDesk_Flexible_Shipping_SaaS_Default_Value_Compute_Handler( $field[ $deafault_commpute ] );
					$settings_field['default'] = $default_compute_handler->compute();
				}
			}
			return $settings_field;
		}

		/**
		 * Prepare settings field select options.
		 *
		 * @param array $field Field.
		 * @param array $settings_field Settings field.
		 *
		 * @return mixed
		 */
		private function prepare_field_select_options( array $field, array $settings_field ) {
			if ( self::TYPE_SELECT === $field[ self::FIELD_TYPE ] || self::TYPE_CUSTOM_SERVICES === $field[ self::FIELD_TYPE ] ) {
				if ( isset( $field[ self::FIELD_SELECT_ADDITIONAL ] ) && isset( $field[ self::FIELD_SELECT_ADDITIONAL ][ self::FIELD_OPTIONS ] ) ) {
					$options = array();
					foreach ( $field[ self::FIELD_SELECT_ADDITIONAL ][ self::FIELD_OPTIONS ] as $option ) {
						if ( ! empty( $option['value'] ) || ! empty( $option['text'] ) ) {
							$options[ $option['value'] ] = $option['text'];
						}
					}
					$settings_field[ self::FIELD_OPTIONS ] = $options;
				}
			}
			return $settings_field;
		}

		/**
		 * Prepare settings field custom services options.
		 *
		 * @param array $field Field.
		 * @param array $settings_field Settings field.
		 *
		 * @return mixed
		 */
		private function prepare_field_custom_services_options( array $field, array $settings_field ) {
			if ( self::TYPE_SELECT === $field[ self::FIELD_TYPE ] || self::TYPE_CUSTOM_SERVICES === $field[ self::FIELD_TYPE ] ) {
				if ( isset( $field[ self::FIELD_SELECT_ADDITIONAL ] ) && isset( $field[ self::FIELD_SELECT_ADDITIONAL ][ self::FIELD_OPTIONS ] ) ) {
					$services = array();
					foreach ( $field[ self::FIELD_SELECT_ADDITIONAL ][ self::FIELD_OPTIONS ] as $option ) {
						if ( ! empty( $option['value'] ) || ! empty( $option['text'] ) ) {
							$services[ $option['value'] ] = array( 'name' => $option['text'] );
						}
					}
					$settings_field['services'] = $services;
				}
			}
			return $settings_field;
		}

		/**
		 * Prepare settings field description.
		 *
		 * @param array $field Field.
		 * @param array $settings_field Settings field.
		 *
		 * @return mixed
		 */
		private function prepare_field_description( array $field, array $settings_field ) {
			if ( isset( $field[ self::FIELD_DESC ] ) ) {
				$settings_field[ self::FIELD_DESCRIPTION ] = do_shortcode( $field[ self::FIELD_DESC ] );
			}
			if ( isset( $field[ self::FIELD_TOOLTIP ] ) && is_array( $field[ self::FIELD_TOOLTIP ] ) ) {
				$settings_field[ self::FIELD_DESCRIPTION ] = do_shortcode( $field[ self::FIELD_TOOLTIP ][ self::FIELD_DESCRIPTION ] );
				$settings_field[ self::FIELD_DESC_TIP ]    = true;
			}
			return $settings_field;
		}

		/**
		 * Prepare setting field.
		 *
		 * @param array $field Field.
		 *
		 * @return array
		 */
		private function prepare_setting_field( array $field ) {
			$settings_field = array(
				self::FIELD_TYPE              => $field[ self::FIELD_TYPE ],
				self::FIELD_TITLE             => do_shortcode( $field['name'] ),
				self::FIELD_CUSTOM_ATTRIBUTES => array(),
			);
			if ( isset( $field[ self::FIELD_LABEL ] ) ) {
				$settings_field[ self::FIELD_LABEL ] = $field[ self::FIELD_LABEL ];
			}
			if ( 'password' === $field[ self::FIELD_TYPE ] ) {
				$settings_field[ self::FIELD_CUSTOM_ATTRIBUTES ]['autocomplete'] = 'new-password';
			}
			if ( self::TYPE_CHECKBOX !== $field[ self::FIELD_TYPE ] && self::TYPE_SELECT !== $field[ self::FIELD_TYPE ]
				&& isset( $field[ self::FIELD_REQUIRED ] ) && $field[ self::FIELD_REQUIRED ]
			) {
				$settings_field[ self::FIELD_CUSTOM_ATTRIBUTES ][ self::FIELD_REQUIRED ] = self::FIELD_REQUIRED;
			}
			if ( self::TYPE_CUSTOM_SERVICES === $field[ self::FIELD_TYPE ] ) {
				$settings_field = $this->prepare_field_custom_services_options( $field, $settings_field );
			}
			$settings_field = $this->prepare_default_field_value( $field, $settings_field );
			$settings_field = $this->prepare_field_select_options( $field, $settings_field );
			$settings_field = $this->prepare_field_description( $field, $settings_field );
			return $settings_field;
		}


		/**
		 * Prepare settings for platform.
		 *
		 * @param array $settings Settings.
		 *
		 * @return array
		 */
		public function prepare_settings_for_platform( array $settings ) {
			$settings_for_platform = array();
			foreach ( $this->settings_fields as $section ) {
				foreach ( $section['fields'] as $field ) {
					$field_id = $field['id'];
					if ( isset( $field[ self::FIELD_TYPE ] ) && self::FIELD_TYPE_CONNECTION_STATUS !== $field[ self::FIELD_TYPE ] ) {
						if ( isset( $settings[ $field_id ] ) ) {
							$settings_for_platform[ $field_id ] = $settings[ $field_id ];
						} else {
							$settings_for_platform[ $field_id ] = '';
						}
						if ( self::TYPE_CHECKBOX === $field[ self::FIELD_TYPE ] ) {
							$settings_for_platform[ $field_id ] = 'yes' === $settings[ $field_id ];
						}
						if ( 'shipment_settings' === $field[ self::FIELD_TYPE ] ) {
							$shipment_settings = new WPDesk_Flexible_Shipping_SaaS_Shipment_Settings_Field( $field );

							$settings_for_platform = $shipment_settings->add_settings_for_platform( $settings, $settings_for_platform );
						}
					}
				}
			}
			$country_state_field   = WPDesk_Flexible_Shipping_SaaS_Country_State_Field::create_from_serialized_settings( $settings );
			$settings_for_platform = array_merge( $settings_for_platform, $country_state_field->get_as_array() );
			return $settings_for_platform;
		}

		/**
		 * Prepare woocommerce settings fields from fields.
		 *
		 * @param array $fields Fields.
		 *
		 * @return array
		 */
		private function prepare_woocommerce_settings_fields_from_fields( array $fields ) {
			$settings_fields = array();
			foreach ( $fields as $section ) {
				$settings_fields[] = $this->prepare_section_heading_field( $section );
				foreach ( $section['fields'] as $field ) {
					if ( 'shipment_settings' === $field[ self::FIELD_TYPE ] ) {
						$shipment_settings_field = new WPDesk_Flexible_Shipping_SaaS_Shipment_Settings_Field( $field );

						$settings_fields = $shipment_settings_field->add_fields_to_settings( $settings_fields );
					} else {
						$settings_fields[ $field['id'] ] = $this->prepare_setting_field( $field );
					}
				}
			}
			return $settings_fields;
		}

		/**
		 * Get shipping method settings fields.
		 *
		 * @return array
		 */
		public function prepare_shipping_method_settings_fields() {
			return $this->prepare_woocommerce_settings_fields_from_fields( $this->settings_fields );
		}

		/**
		 * Set css class on field.
		 *
		 * @param array $field Settings field.
		 * @param array $show_in Show in.
		 *
		 * @return array
		 */
		private function set_css_class_on_field( array $field, array $show_in ) {
			if ( ! isset( $field['class'] ) ) {
				$field['class'] = '';
			}
			if ( in_array( 'method', $show_in, true ) ) {
				$field['class'] .= ' fs-method';
			}
			if ( in_array( 'live', $show_in, true ) ) {
				$field['class'] .= ' fs-live';
			}
			return $field;
		}

		/**
		 * Prepare rates type field.
		 *
		 * @return array
		 */
		private function prepare_rates_type_field() {
			$rates_type_field = array(
				'title'       => __( 'Rates type', 'flexible-shipping' ),
				'type'        => self::TYPE_SELECT,
				'class'       => 'fs-rates-type fs-method fs-live',
				'options'     => array(
					'custom' => __( 'Custom', 'flexible-shipping' ),
				),
				// Translators: service name.
				'description' => sprintf( __(
					'Choose custom rates to set a custom shipping cost (this method will be the selected %1$s service). Choose live rates if you want to present services and rates for the customer\'s address.',
					'flexible-shipping'
				), $this->get_name() ),
				'desc_tip'    => true,
			);
			if ( $this->supports( 'supportsShipmentLiveRates' ) ) {
				$rates_type_field['options'] = array(
					'live'   => __( 'Live', 'flexible-shipping' ),
					'custom' => __( 'Custom', 'flexible-shipping' ),
				);
				$rates_type_field['default'] = 'live';
			}
			return $rates_type_field;
		}

		/**
		 * Prepare custom services field.
		 *
		 * @param string $field_id Field Id.
		 * @param array  $field Field.
		 * @param array  $show_in Show in.
		 *
		 * @return array
		 */
		private function prepare_custom_services_field( $field_id, $field, $show_in ) {
			$custom_services = array(
				'title'                 => __( 'Custom services', 'flexible-shipping' ),
				self::FIELD_TYPE        => self::TYPE_CHECKBOX,
				self::FIELD_DESC_TIP    => true,
				self::FIELD_DESCRIPTION => $field[ self::FIELD_TOOLTIP ][ self::FIELD_DESCRIPTION ],
				'label'                 => __( 'Enable services custom settings', 'flexible-shipping' ),
			);
			$custom_services = $this->set_css_class_on_field( $custom_services, $show_in );

			$custom_services['custom_attributes'] = array( 'data-dependent-field-id' => 'woocommerce_flexible_shipping_' . $field_id );

			$custom_services = $this->prepare_default_field_value( $field, $custom_services );

			return $custom_services;
		}

		/**
		 * Prepare FS integration settings fields.
		 *
		 * @param string $field_name_prefix Field name prefix.
		 * @param array  $zone_targets Zone targets.
		 * @return array
		 */
		public function prepare_flexible_shipping_integration_settings_fields( $field_name_prefix, array $zone_targets ) {
			$settings = array( $field_name_prefix . 'rates_type' => $this->prepare_rates_type_field() );
			$fields   = $this->get_fields_for_targets( $zone_targets );
			foreach ( $fields as $section ) {
				if ( in_array( 'method', $section['show-in'], true ) || in_array( 'live', $section['show-in'], true ) ) {
					foreach ( $section['fields'] as $field ) {
						$field_id = $field_name_prefix . $field['id'];
						if ( self::TYPE_CUSTOM_SERVICES === $field['type'] ) {
							$field_id = $field_id . '_custom_services';
							$settings[ $field_name_prefix . self::TYPE_CUSTOM_SERVICES ] = $this->prepare_custom_services_field(
								$field_id,
								$field,
								$section['show-in']
							);
						}
						$settings_field = $this->prepare_setting_field( $field );
						$settings_field = $this->set_css_class_on_field( $settings_field, $section['show-in'] );

						$settings[ $field_id ] = $settings_field;
					}
				}
			}
			if ( $this->supports( 'supportsShipmentLiveRates' ) ) {
				$settings_field = array(
					'title'             => __( 'Fallback', 'flexible-shipping' ),
					'label'             => __( 'Enable fallback', 'flexible-shipping' ),
					'type'              => 'checkbox',
					'description'       => __( 'Enable to offer flat rate cost for shipping so that the user can still checkout, if API returns no matching rates.', 'flexible-shipping' ),
					'desc_tip'          => true,
					'custom_attributes' => array(
						'data-dependent-row-class' => 'flexible_shipping_method_rules',
						'data-dependent-field-id'  => 'woocommerce_flexible_shipping_method_calculation_method',
					),
				);
				$settings_field = $this->set_css_class_on_field( $settings_field, array( 'live' ) );

				$settings[ $field_name_prefix . 'fallback' ] = $settings_field;
			}
			return $settings;
		}


		/**
		 * Cast raw service settings to valid types.
		 *
		 * @param array $settings Settings values.
		 *
		 * @return array Settings values with casting.
		 */
		public function cast_raw_service_settings_to_valid_types( array $settings ) {
			foreach ( $this->settings_fields_values['properties'] as $key => $type ) {
				if ( 'integer' === $type[ self::FIELD_TYPE ] ) {
					$settings[ $key ] = (int) $settings[ $key ];
				}
			}
			return $settings;
		}

		/**
		 * Get fields for zone targets.
		 *
		 * @param array $zone_targets Target countries.
		 *
		 * @return array
		 * @throws WPDesk_Flexible_Shipping_SaaS_Bad_Credentials_Exception Exception.
		 * @throws WPDesk_Flexible_Shipping_SaaS_Service_Settings_Not_Found Exception.
		 * @throws WPDesk_Flexible_Shipping_SaaS_Maintenance_Mode_Exception Exception.
		 */
		public function get_fields_for_targets( array $zone_targets ) {
			$platform        = $this->saas_connection->get_platform();
			$fields_response = $platform->requestGetFields( $this->get_id(), $zone_targets );
			if ( $fields_response->isBadCredentials() ) {
				throw new WPDesk_Flexible_Shipping_SaaS_Bad_Credentials_Exception( $this->saas_platform_links );
			} elseif ( $fields_response->isMaintenance() ) {
				throw new WPDesk_Flexible_Shipping_SaaS_Maintenance_Mode_Exception( $fields_response );
			} elseif ( $fields_response->isNotExists() ) {
				throw new WPDesk_Flexible_Shipping_SaaS_Service_Settings_Not_Found();
			}
			return $fields_response->getFields();
		}

		/**
		 * Set shipping method.
		 *
		 * @param WPDesk_Flexible_Shipping_SaaS_Shipping_Service_Settings $shipping_method Shipping method.
		 */
		public function set_shipping_method( WPDesk_Flexible_Shipping_SaaS_Shipping_Service_Settings $shipping_method ) {
			$this->shipping_method = $shipping_method;
		}

		/**
		 * Get option from shiping method.
		 *
		 * @param string $option_name Option name.
		 *
		 * @return string
		 */
		public function get_option_from_shipping_method( $option_name ) {
			$option_value = '';
			if ( ! empty( $this->shipping_method ) ) {
				return $this->shipping_method->get_option( $option_name, '' );
			}
			return $option_value;
		}

		/**
		 * Is auto selected in shipment_settings_auto_create option?
		 *
		 * @return bool
		 */
		public function is_auto_shipment_settings_auto_create() {
			return 'auto' === $this->get_option_from_shipping_method( 'shipment_settings_auto_create' );
		}

		/**
		 * Is manual selected in shipment_settings_auto_create option?
		 *
		 * @return bool
		 */
		public function is_manual_shipment_settings_auto_create() {
			return 'manual' === $this->get_option_from_shipping_method( 'shipment_settings_auto_create' );
		}

		/**
		 * Is order status set in shipment_settings_order_status option?
		 *
		 * @param string $order_status Order status.
		 *
		 * @return bool
		 */
		public function is_order_status_shipment_settings_order_status( $order_status ) {
			return 'wc-' . $order_status === $this->get_option_from_shipping_method( 'shipment_settings_order_status' );
		}

		/**
		 * Is shipment_settings_complete_order option checked?
		 *
		 * @return bool
		 */
		public function is_shipment_settings_complete_order() {
			return 'yes' === $this->get_option_from_shipping_method( 'shipment_settings_complete_order' );
		}

		/**
		 * Get settings URL.
		 *
		 * @return string
		 */
		public function get_settings_url() {
			return admin_url( self::WOOCOMMERCE_SETTINGS_SHIPPING_URL . '&section=' . $this->integration_id );
		}

		/**
		 * Prepare settings not found message.
		 *
		 * @return string
		 */
		public function prepare_settings_not_found_message() {
			return sprintf(
				// Translators: links to settings.
				__( 'You need to configure the service. %1$sGo to %2$s settings.%3$s', 'flexible-shipping' ),
				'<a href="' . $this->get_settings_url() . '"">',
				$this->get_name(),
				'</a>'
			);
		}

	}

}
