<?php

namespace WPDesk\Logger;

use Monolog\Handler\AbstractHandler;
use Monolog\Handler\NullHandler;
use Monolog\Logger;
use Monolog\Registry;
use Monolog\ErrorHandler;
use Monolog\Handler\StreamHandler;
use Psr\Log\LogLevel;
use WPDesk\Logger\Filter\BooleanFilter;
use WPDesk\Logger\WC\WooCommerceCapture;

/**
 * Manages and facilitates creation of logger
 *
 * @package WPDesk\Logger
 */
class WPDeskLoggerFactory extends BasicLoggerFactory
{
    const DEFAULT_LOGGER_CHANNEL_NAME = 'wpdesk';

    /** @var string Log to file when level is */
    const LEVEL_WPDESK_FILE = LogLevel::DEBUG;

    /** @var string Log to wc logger when level is */
    const LEVEL_WC = LogLevel::ERROR;

    /**
     * Remove static instances. In general should be use only testing purposes.
     *
     * @param string $name Name of the logger
     */
    public static function tearDown($name = self::DEFAULT_LOGGER_CHANNEL_NAME)
    {
        if (Registry::hasLogger($name)) {
            Registry::removeLogger($name);
        }
    }

    /**
     * Disable logger. Still exists but logs won't be saved
     *
     * @param string $name Name of the logger
     */
    public function disableLog($name = self::DEFAULT_LOGGER_CHANNEL_NAME)
    {
        if (!Registry::hasLogger($name)) {
            $this->createWPDeskLogger($name);
        }
        /** @var Logger $logger */
        $logger = Registry::getInstance($name);
        $this->removeAllHandlers($logger);
    }

    /**
     * Creates default WPDesk logger.
     *
     * Requirements:
     * - get_option, add/remove_action, add/remove filter and WP_CONTENT_DIR should be available for logger.
     *
     * Assumptions:
     * - logger is actively working when 'wpdesk_helper_options' has 'debug_log' set to '1';
     * - fatal errors, exception and standard errors are recorded but in a transparent way;
     * - WooCommerce logger is captured and returns this logger;
     * - logs are still correctly written to WooCommerce subsystem in a transparent way;
     * - all recorded errors are written to WPDesk file.
     *
     * @param string $name Name of the logger
     * @return Logger
     */
    public function createWPDeskLogger($name = self::DEFAULT_LOGGER_CHANNEL_NAME)
    {
        if (Registry::hasLogger($name)) {
            return Registry::getInstance($name);
        }
        $logger = $this->createLogger($name);
        $this->captureWooCommerce($logger);
        $this->captureErrorHandle($logger);
        try {
            $this->pushFileHandle($name, $logger);
        } catch (\InvalidArgumentException $e) {
            $logger->emergency('File log could not be created - invalid filename.');
        } catch (\Exception $e) {
            $logger->emergency('File log could not be written.');
        }

        return $logger;
    }

    /**
     * Capture WooCommerce and add handle
     *
     * @param Logger $logger
     */
    private function captureWooCommerce(Logger $logger)
    {
        if (!defined('WC_LOG_THRESHOLD')) {
            define('WC_LOG_THRESHOLD', self::LEVEL_WC);
        }

        $wcIntegration = new WooCommerceCapture($logger);
        $wcIntegration->captureWcLogger();
    }

    /**
     * Add WordPress(standard PHP) errors handle
     *
     * @param Logger $logger
     */
    private function captureErrorHandle(Logger $logger)
    {
        $errorHandler = new ErrorHandler($logger);
        $errorHandler->registerErrorHandler();
    }

    /**
     * Add WPDesk log file handle
     *
     * @param Logger $logger
     * @param string $name Name of the logger
     *
     * @throws \Exception                If a missing directory is not buildable
     * @throws \InvalidArgumentException If stream is not a resource or string
     */
    private function pushFileHandle($name, Logger $logger)
    {
        $filename = $this->getFileName($name);
        $logger->pushHandler(new StreamHandler($filename, self::LEVEL_WPDESK_FILE));
    }

    /**
     * Get filename old way
     *
     * @deprecated
     */
    public function getWPDeskFileName()
    {
        return $this->getFileName();
    }

    /**
     * Returns WPDesk filename with path.
     *
     * @param string $name Name of the logger
     *
     * @return string
     */
    public function getFileName($name = self::DEFAULT_LOGGER_CHANNEL_NAME)
    {
        return WP_CONTENT_DIR . '/uploads/logs/' . $name . '_debug.log';
    }

    /**
     * Removes all handlers from logger
     *
     * @param Logger $logger
     *
     * @return void
     */
    private function removeAllHandlers(Logger $logger)
    {
        try {
            while (true) {
                $logger->popHandler();
            }
        } catch (\LogicException $e) {
            $logger->pushHandler(new NullHandler());
        }
    }

    /**
     * is WPDesk file log is working(writable, exists, connected).
     * @param string $name Name of the logger
     *
     * @return bool
     */
    public function isLogWorking($name = self::DEFAULT_LOGGER_CHANNEL_NAME)
    {
        return Registry::hasLogger($name);
    }
}
