<?php

use Monolog\Handler\AbstractHandler;
use Monolog\Logger;
use PHPUnit\Framework\Error\Notice;
use WPDesk\Logger\WC\WooCommerceMonologPlugin;
use WPDesk\Logger\WPDeskLoggerFactory;

class TestWPDeskLoggerFactory extends WP_UnitTestCase
{
    const LOGGER_NAME = 'somerandomname';

    public function tearDown()
    {
        parent::tearDown();
        WPDeskLoggerFactory::tearDown(self::LOGGER_NAME);
    }

    public function testCanCreateLogger()
    {
        $factory = new WPDeskLoggerFactory();
        $this->assertInstanceOf(Logger::class, $factory->createWPDeskLogger(self::LOGGER_NAME), "Logger should be created");
        $this->assertTrue($factory->isLogWorking(self::LOGGER_NAME));
    }

    public function testWCIsNotCorruptedByOtherTests()
    {
        $this->assertInstanceOf(WC_Logger::class, wc_get_logger(), "Logger should be NOT captured by default.");
    }

    public function testWCLoggingIsCapturedByOurLogs()
    {
        (new WPDeskLoggerFactory())->createWPDeskLogger();
        $this->assertInstanceOf(WooCommerceMonologPlugin::class, wc_get_logger(), "Logger should be captured.");
    }

    public function testOurLogsGetAllErrors()
    {
        $someMessage = 'whatever';
        $errorType = E_USER_NOTICE;

        $logger = (new WPDeskLoggerFactory())->createWPDeskLogger(self::LOGGER_NAME);
        $logger->pushHandler($this->prepareListenHandleThatIsWaitingForMessage('E_USER_NOTICE: ' . $someMessage));
        $this->expectException(Notice::class);
        trigger_error($someMessage, $errorType);
    }

    /**
     * Prepares listener that check if logger gets sent message
     *
     * @param $message
     * @return AbstractHandler
     */
    private function prepareListenHandleThatIsWaitingForMessage($message)
    {
        $listenHandle = $this->createMock(AbstractHandler::class);

        $listenHandle
            ->expects($this->atLeastOnce())
            ->method('handle')
            ->with($this->callback(function ($record) use ($message) {
                $this->assertEquals($message, $record['message'], "Monolog should get message sent to logger");
                return $record['message'] === $message;
            }))
            ->willReturn(true);

        $listenHandle
            ->expects($this->atLeastOnce())
            ->method('isHandling')
            ->willReturn(true);

        /** @var $listenHandle AbstractHandler */
        return $listenHandle;
    }

    public function testOurLogsGetAllMessagesLoggedToWC()
    {
        $someMessage = 'whatever';
        $logger = (new WPDeskLoggerFactory())->createWPDeskLogger(self::LOGGER_NAME);
        $logger->pushHandler($this->prepareListenHandleThatIsWaitingForMessage($someMessage));

        wc_get_logger()->debug($someMessage);
    }

    public function testLoggerWorksAndCanLogInGeneral()
    {
        $someMessage = 'whatever';
        $logger = (new WPDeskLoggerFactory())->createWPDeskLogger(self::LOGGER_NAME);
        $logger->pushHandler($this->prepareListenHandleThatIsWaitingForMessage($someMessage));

        $logger->debug($someMessage);
    }

    public function testAllLoggedMessagesAreWrittenToWPDeskFile()
    {
        $someMessage = 'whatever';

        $factory = new WPDeskLoggerFactory();
        $logFilename = $factory->getFileName(self::LOGGER_NAME);

        @unlink($logFilename);
        $this->assertFileNotExists($logFilename);

        $logger = $factory->createWPDeskLogger(self::LOGGER_NAME);
        $logger->debug($someMessage);
        $this->assertFileExists($logFilename);
    }

    public function testAllLoggedMessagesAreWrittenToWC()
    {
        $mockWcLogger = $this->createMock(WC_Logger::class);
        $mockWcLogger
            ->expects($this->atLeastOnce())
            ->method('log')
            ->willReturn(true);

        add_filter('woocommerce_logging_class', function () use ($mockWcLogger) {
            return $mockWcLogger;
        }, 0, 100);

        $someMessage = 'whatever';
        $logger = (new WPDeskLoggerFactory())->createWPDeskLogger(self::LOGGER_NAME);

        $logger->debug($someMessage);
    }

    public function testDisable() {
        $factory = new WPDeskLoggerFactory();
        $logger = $factory->createWPDeskLogger(self::LOGGER_NAME);
        $factory->disableLog(self::LOGGER_NAME);

        $listenHandle = $this->createMock(AbstractHandler::class);

        $listenHandle
            ->expects($this->never())
            ->method('handle');

        $logger->pushHandler($listenHandle);

        $logger->debug('whatever');
    }
}

