<?php

namespace WPDesk\SaasPlatformClient\ApiClient;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;
use WPDesk\SaasPlatformClient\Cache\CacheDispatcher;
use WPDesk\SaasPlatformClient\Cache\CacheItemCreator;
use WPDesk\SaasPlatformClient\Cache\CacheItemVerifier;
use WPDesk\SaasPlatformClient\HttpClient\Curl\CurlException;
use WPDesk\SaasPlatformClient\HttpClient\HttpClient;
use WPDesk\SaasPlatformClient\HttpClient\HttpClientResponse;
use WPDesk\SaasPlatformClient\Model\Status;
use WPDesk\SaasPlatformClient\Platform;
use WPDesk\SaasPlatformClient\Request\Request;
use WPDesk\SaasPlatformClient\Request\Status\GetStatusRequest;
use WPDesk\SaasPlatformClient\Response\ProtectedResponse;
use WPDesk\SaasPlatformClient\Response\RawResponse;
use WPDesk\SaasPlatformClient\Response\Response;
use WPDesk\SaasPlatformClient\Response\Status\GetStatusResponse;
use WPDesk\SaasPlatformClient\Serializer\Serializer;

class CachedClient implements Client, CacheItemCreator, CacheItemVerifier
{

    const OPTION_FS_SAAS_PLATFORM_VERSION = 'fs-saas-platform-version';

    /** @var Client */
    private $client;

    /** @var CacheInterface */
    private $cache;

    /**
     * @var CacheDispatcher
     */
    private $cacheDispatcher;

    /**
     * CachedClient constructor.
     * @param Client $decorated Decorated client
     * @param CacheInterface $cache
     */
    public function __construct(Client $decorated, CacheInterface $cache)
    {
        $this->client = $decorated;
        $this->cache = $cache;
        $this->cacheDispatcher = new CacheDispatcher($cache, [new RequestCacheInfoResolver()]);
    }

    /**
     * Create item to cache.
     *
     * @param Request $request
     * @return Response
     */
    public function createCacheItem($request)
    {
        return $this->client->sendRequest($request);
    }

    /**
     * Verify cache item.
     *
     * @param $object
     * @return Response;
     */
    public function getVerifiedItemOrNull($object)
    {
        if ($object instanceof Response) {
            return $object;
        }
        return null;
    }

    /**
     * Send request.
     *
     * @param Request $request
     * @return mixed|Response
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function sendRequest(Request $request)
    {
        $this->maybeClearCacheBeforeRequest($request);
        return $this->cacheDispatcher->dispatch($request, $this, $this);
    }

    /**
     * Maybe clear cache before request.
     *
     * @param Request $request
     *
     * @return bool
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    private function maybeClearCacheBeforeRequest(Request $request)
    {
        if (! $request instanceof GetStatusRequest) {
            try {
                $statusResponse = $this->sendStatusRequest();
                $status         = $statusResponse->getStatus();
                if (!$this->isLocalPlatformVersionCurrent($status)) {
                    $this->cache->clear();
                    update_option(self::OPTION_FS_SAAS_PLATFORM_VERSION, $status->getPlatformVersion());
                }
            } catch (\Exception $e) {
                $this->cache->clear();
            }
        }
    }

    /**
     * Send status request.
     *
     * @return GetStatusResponse
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    private function sendStatusRequest()
    {
        return new GetStatusResponse(new ProtectedResponse($this->sendRequest(new GetStatusRequest())));
    }

    /**
     * Is local platform version current?
     *
     * @param Status $status
     *
     * @return bool
     */
    private function isLocalPlatformVersionCurrent(Status $status)
    {
        if ($status->getPlatformVersion() === get_option(self::OPTION_FS_SAAS_PLATFORM_VERSION, '-1')) {
            return true;
        }
        return false;
    }

    /**
     * @return HttpClient
     */
    public function getHttpClient()
    {
        return $this->client->getHttpClient();
    }

    /**
     * @param HttpClient $client
     * @return mixed
     */
    public function setHttpClient(HttpClient $client)
    {
        return $this->client->setHttpClient($client);
    }

    /**
     * @return Serializer
     */
    public function getSerializer()
    {
        return $this->client->getSerializer();
    }

    /**
     * @return string
     */
    public function getApiUrl()
    {
        return $this->client->getApiUrl();
    }

}