<?php

/*
Copyright (C) 2016-2017 Pimwick, LLC

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

defined( 'ABSPATH' ) or exit;

if ( ! class_exists( 'PW_Gift_Cards_Purchasing' ) ) :

final class PW_Gift_Cards_Purchasing {

    function __construct() {

        add_filter( 'woocommerce_get_price_html', array( $this, 'woocommerce_get_price_html' ), 10, 2 );
        add_filter( 'woocommerce_dropdown_variation_attribute_options_args', array( $this, 'woocommerce_dropdown_variation_attribute_options_args' ) );
        add_filter( 'woocommerce_add_to_cart_handler', array( $this, 'woocommerce_add_to_cart_handler' ), 10 , 2 );
        add_filter( 'woocommerce_add_cart_item', array( $this, 'woocommerce_add_cart_item' ) );
        add_filter( 'woocommerce_add_cart_item_data', array( $this, 'woocommerce_add_cart_item_data' ), 10, 3 );
        add_filter( 'woocommerce_get_cart_item_from_session', array( $this, 'woocommerce_get_cart_item_from_session' ), 10, 2 );
        add_filter( 'woocommerce_get_item_data', array( $this, 'woocommerce_get_item_data' ), 10, 2 );
        add_filter( 'woocommerce_cart_item_permalink', array( $this, 'woocommerce_cart_item_permalink' ), 10, 3 );
        add_filter( 'woocommerce_checkout_create_order_line_item', array( $this, 'woocommerce_checkout_create_order_line_item' ), 10, 4 );
        add_filter( 'woocommerce_order_status_processing', array( $this, 'woocommerce_order_status_processing' ), 11, 2 );
        add_filter( 'woocommerce_order_status_completed', array( $this, 'woocommerce_order_status_completed' ), 11, 2 );
        add_filter( 'woocommerce_order_status_cancelled', array( $this, 'woocommerce_order_status_cancelled' ), 11, 2 );
        add_filter( 'woocommerce_order_status_refunded', array( $this, 'woocommerce_order_status_refunded' ), 11, 2 );
        add_filter( 'woocommerce_order_item_permalink', array( $this, 'woocommerce_order_item_permalink' ), 10, 3 );
        add_filter( 'woocommerce_order_item_display_meta_key', array( $this, 'woocommerce_order_item_display_meta_key' ), 10, 3 );
        add_action( 'woocommerce_order_again_cart_item_data', array( $this, 'woocommerce_order_again_cart_item_data' ), 10, 3 );
        add_filter( 'woocommerce_product_get_price', array( $this, 'woocommerce_product_get_price' ), 10, 2 );
    }

    function woocommerce_get_price_html( $price, $product ) {
        if ( is_a( $product, 'WC_Product_Variation' ) && empty( $product->get_price() ) ) {
            $product_id = $product->get_parent_id();
            $parent_product = wc_get_product( $product_id );
            if ( is_a( $parent_product, 'WC_Product_PW_Gift_Card' ) ) {
                $price = '';
            }
        }

        return $price;
    }

    function woocommerce_dropdown_variation_attribute_options_args( $args ) {
        if ( $args['product'] && is_a( $args['product'], 'WC_Product_PW_Gift_Card' ) ) {
            $args['show_option_none'] = __( 'Choose an amount', 'pw-woocommerce-gift-cards' );
        }

        return $args;
    }

    function woocommerce_add_to_cart_handler( $product_type, $product ) {
        if ( $product_type == PWGC_PRODUCT_TYPE_SLUG ) {
            return 'variable';
        } else {
            return $product_type;
        }
    }

    function woocommerce_add_cart_item( $cart_item ) {
        global $pw_gift_cards;

        $product_id = absint( $cart_item['product_id'] );
        $variation_id = absint( $cart_item['variation_id'] );

        if ( isset( $cart_item[ PWGC_TO_META_KEY ] ) && !empty( $cart_item[ PWGC_TO_META_KEY ] ) ) {
            $recipients = preg_split('/[\s,]+/', $cart_item[ PWGC_TO_META_KEY ], PWGC_RECIPIENT_LIMIT, PREG_SPLIT_NO_EMPTY);
            if ( count( $recipients ) > 1 ) {
                $cart_item['quantity'] = count( $recipients );
            }
        }

        return $cart_item;
    }

    function woocommerce_add_cart_item_data( $cart_item_data, $product_id, $variation_id ) {
        global $pw_gift_cards;

        foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
            if ( isset( $_POST[ $key ] ) ) {
                if ( $key == PWGC_MESSAGE_META_KEY ) {
                    $cart_item_data[ $key ] = sanitize_textarea_field( stripslashes( $_POST[ $key ] ) );
                } else {
                    $cart_item_data[ $key ] = sanitize_text_field( stripslashes( $_POST[ $key ] ) );
                }
            }
        }

        return $cart_item_data;
    }

    function woocommerce_get_cart_item_from_session( $cart_item, $values ) {
        global $pw_gift_cards;

        foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
            if ( isset( $values[ $key ] ) ) {
                $cart_item[ $key ] = $values[ $key ];
            }
        }

        return $cart_item;
    }

    function woocommerce_get_item_data( $item_data, $cart_item ) {
        global $pw_gift_cards;

        foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
            if ( isset( $cart_item[ $key ] ) ) {
                $value = $cart_item[ $key ];
                if ( !empty( $value ) ) {
                    $item_data[] = array(
                        'key' => $display,
                        'value' => $value
                    );
                }
            }
        }

        return $item_data;
    }

    function woocommerce_cart_item_permalink( $product_permalink, $cart_item, $cart_item_key ) {
        global $pw_gift_cards;

        if ( !empty( $product_permalink ) ) {
            foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
                if ( strpos( $product_permalink, $key ) === false && isset( $cart_item[ $key ] ) ) {
                    $product_permalink = add_query_arg( $key, urlencode( nl2br( $cart_item[ $key ] ) ), $product_permalink );
                }
            }
        }

        return $product_permalink;
    }

    function woocommerce_checkout_create_order_line_item( $order_item, $cart_item_key, $cart_item, $order ) {
        global $pw_gift_cards;

        $product = wc_get_product( $order_item->get_product_id() );
        if ( is_a( $product, 'WC_Product_PW_Gift_Card' ) ) {
            foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
                if ( $key == PWGC_AMOUNT_META_KEY ) {
                    $order_item->add_meta_data( $key, $cart_item['data']->get_regular_price() );
                } else if ( isset( $cart_item[ $key ] ) ) {
                    $order_item->add_meta_data( $key, $cart_item[ $key ] );
                }
            }
        }
    }

    function woocommerce_order_status_processing( $order_id, $order ) {
        $this->add_gift_cards_to_order( $order_id, $order, "order_id: $order_id processing" );
    }

    function woocommerce_order_status_completed( $order_id, $order ) {
        $this->add_gift_cards_to_order( $order_id, $order, "order_id: $order_id completed" );
    }

    function woocommerce_order_status_cancelled( $order_id, $order ) {
        $this->deactivate_gift_cards_from_order( $order_id, $order, "order_id: $order_id cancelled" );
    }

    function woocommerce_order_status_refunded( $order_id, $order ) {
        $this->deactivate_gift_cards_from_order( $order_id, $order, "order_id: $order_id refunded" );
    }

    function add_gift_cards_to_order( $order_id, $order, $note ) {

        $create_note = sprintf( __( 'Order %s purchased by %s %s', 'pw-woocommerce-gift-cards' ), $order->get_id(), $order->get_billing_first_name(), $order->get_billing_last_name() );

        foreach ( $order->get_items( 'line_item' ) as $order_item_id => $order_item ) {

            // Make sure we have a quantity (should always be true, right? Oh well, prevents a divide-by-zero error just in case).
            if ( $order_item->get_quantity() <= 0 ) {
                continue;
            }

            // Get the product.
            $product_id = absint( $order_item['product_id'] );
            if ( !( $product = wc_get_product( $product_id ) ) ) {
                continue;
            }

            // We're only interested in these guys.
            if ( !is_a( $product, 'WC_Product_PW_Gift_Card' ) ) {
                continue;
            }

            // Grab the Variation, otherwise there will be trouble.
            $variation_id = absint( $order_item['variation_id'] );
            if ( !( $variation = wc_get_product( $variation_id ) ) ) {
                wp_die( __( 'Unable to retrieve variation ', 'pw-woocommerce-gift-cards' ) . $variation_id );
            }

            $credit_amount = wc_get_order_item_meta( $order_item->get_id(), PWGC_AMOUNT_META_KEY );
            if ( !is_numeric( $credit_amount ) || empty( $credit_amount ) ) {

                // Previously we didn't store the PWGC_AMOUNT_META_KEY so we need to calculate based on purchase price.
                $credit_amount = round( $order_item->get_subtotal() / $order_item->get_quantity(), wc_get_price_decimals() );
                if ( !is_numeric( $credit_amount ) || empty( $credit_amount ) ) {
                    continue;
                }
            }

            $item_note = $note . ", order_item_id: $order_item_id";

            // Create a gift card for each quantity ordered.
            $gift_card_numbers = (array) wc_get_order_item_meta( $order_item_id, PWGC_GIFT_CARD_NUMBER_META_KEY, false );

            // Make sure any existing gift cards are activated.
            foreach ( $gift_card_numbers as $gift_card_number ) {
                $gift_card = new PW_Gift_Card( $gift_card_number );
                $gift_card->reactivate( $item_note );
            }

            // Create any new/missing gift cards.
            for ( $x = count( $gift_card_numbers ); $x < $order_item['quantity']; $x++ ) {

                $gift_card = PW_Gift_Card::create_card( $create_note );
                $gift_card->credit( $credit_amount, $item_note );

                wc_add_order_item_meta( $order_item_id, PWGC_GIFT_CARD_NUMBER_META_KEY, $gift_card->get_number() );
            }
        }

        do_action( 'pw_gift_cards_send_emails', $order_id );
    }

    function deactivate_gift_cards_from_order( $order_id, $order, $note ) {
        foreach ( $order->get_items( 'line_item' ) as $order_item_id => $order_item ) {
            $item_note = $note . ", order_item_id: $order_item_id";

            $gift_card_numbers = (array) wc_get_order_item_meta( $order_item_id, PWGC_GIFT_CARD_NUMBER_META_KEY, false );
            foreach ( $gift_card_numbers as $gift_card_number ) {
                $gift_card = new PW_Gift_Card( $gift_card_number );
                $gift_card->deactivate( $item_note );
            }
        }
    }

    function woocommerce_order_item_permalink( $product_permalink, $order_item, $order ) {
        global $pw_gift_cards;

        if ( !empty( $product_permalink ) ) {
            $product = wc_get_product( $order_item->get_product_id() );
            if ( is_a( $product, 'WC_Product_PW_Gift_Card' ) ) {

                foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
                    if ( strpos( $product_permalink, $key ) === false && isset( $order_item[ $key ] ) ) {
                        $product_permalink = add_query_arg( $key, urlencode( $order_item[ $key ] ), $product_permalink );
                    }
                }
            }
        }

        return $product_permalink;
    }

    function woocommerce_order_item_display_meta_key( $display_key, $meta_data, $order_item ) {
        switch ( $display_key ) {
            case PWGC_GIFT_CARD_NUMBER_META_KEY:
                $display_key = PWGC_GIFT_CARD_NUMBER_META_DISPLAY_NAME;

                $gift_card = new PW_Gift_Card( $meta_data->value );
                if ( !$gift_card->get_active() ) {
                    $display_key .= __( ' (inactive)', 'pw-woocommerce-gift-cards' );
                }
            break;
        }

        return $display_key;
    }

    function woocommerce_order_again_cart_item_data( $cart_item_data, $order_item, $order ) {
        global $pw_gift_cards;

        foreach ( $pw_gift_cards->gift_card_meta as $key => $display ) {
            if ( isset( $order_item[ $key ] ) ) {
                if ( $key == PWGC_MESSAGE_META_KEY ) {
                    $cart_item_data[ $key ] = sanitize_textarea_field( stripslashes( $order_item[ $key ] ) );
                } else {
                    $cart_item_data[ $key ] = sanitize_text_field( stripslashes( $order_item[ $key ] ) );
                }
            }
        }

        return $cart_item_data;
    }

    function woocommerce_product_get_price( $value, $product ) {
        if ( is_a( $product, 'WC_Product_PW_Gift_Card' ) && '' === $value ) {
            return '0';
        } else {
            return $value;
        }
    }
}

global $pw_gift_cards_purchasing;
$pw_gift_cards_purchasing = new PW_Gift_Cards_Purchasing();

endif;
